/**
 * enum.h
 *
 *      Copyright (c) 2025 CHBXUE. All rights reserved.
 *
 * THIS FILE AND THE SOFTWARE CONTAINED HEREIN IS PROVIDED 'AS IS' AND
 * COMES WITH NO WARRANTIES OF ANY KIND.
 */

#ifndef LIB_SPREADSHEET_ENUM_H
#define LIB_SPREADSHEET_ENUM_H


#if !defined(COLORREF)
    typedef unsigned long COLORREF;
#endif


// defines the supported modes for performing calculations on workbook data.
enum CalcTypeEnum
{
    CALCTYPE_AUTO,
    CALCTYPE_MANUAL,
    CALCTYPE_AUTONOTABLE
};

enum SheetTypeEnum
{
    SHEETTYPE_UNKNOWN,
    SHEETTYPE_WORKSHEET,
    SHEETTYPE_CHARTSHEET,
    SHEETTYPE_DIALOGSHEET
};

enum SheetStateEnum
{
    SHEETSTATE_VISIBLE,
    SHEETSTATE_HIDDEN,
    SHEETSTATE_VERYHIDDEN
};

enum SheetProtectEnum {
    PROTECT_SELECT_LOCKED_CELLS   = 0x00000001,
    PROTECT_SELECT_UNLOCKED_CELLS = 0x00000002,
    PROTECT_OBJECTS               = 0x00000004,
    PROTECT_SCENARIOS             = 0x00000008,
    PROTECT_FORMAT_CELLS          = 0x00000010,
    PROTECT_FORMAT_COLUMNS        = 0x00000020,
    PROTECT_FORMAT_ROWS           = 0x00000040,
    PROTECT_INSERT_COLUMNS        = 0x00000080,
    PROTECT_INSERT_ROWS           = 0x00000100,
    PROTECT_INSERT_HYPERLINKS     = 0x00000200,
    PROTECT_DELETE_COLUMNS        = 0x00000400,
    PROTECT_DELETE_ROWS           = 0x00000800,
    PROTECT_SORT                  = 0x00001000,
    PROTECT_AUTOFILTER            = 0x00002000,
    PROTECT_PIVOTTABLES           = 0x00004000,

    PROTECT_ALL                   = 0x00007FFF,
    PROTECT_DEFAULT               = 0x00000FFF,
};

enum CellTypeEnum
{
    CELLTYPE_NONE       = -1,
    CELLTYPE_BLANK      =  0,
    CELLTYPE_NUMBER,
    CELLTYPE_STRING,
    CELLTYPE_BOOLEAN,
    CELLTYPE_DATETIME,
    CELLTYPE_ERROR
};

enum ErrorCodeEnum
{
    ERRCODE_UNKNOWN      = -1,
    ERRCODE_NONE         = 0,
    ERRCODE_NULL         = 1,
    ERRCODE_DIV_0        = 2,
    ERRCODE_VALUE        = 3,
    ERRCODE_REF          = 4,
    ERRCODE_NAME         = 5,
    ERRCODE_NUM          = 6,
    ERRCODE_NA           = 7,
    ERRCODE_GETTING_DATA = 8,
    ERRCODE_SPILL,
};

// List of built-in number formats
/*
    ID   formatCode
-----------------------------------
    0     General
    1     0
    2     0.00
    3     #,##0
    4     #,##0.00
    9     0%
    10    0.00%
    11    0.00E+00
    12    # ?/?
    13    # ??/??
    14    mm-dd-yy
    15    d-mmm-yy
    16    d-mmm
    17    mmm-yy
    18    h:mm AM/PM
    19    h:mm:ss AM/PM
    20    h:mm
    21    h:mm:ss
    22    m/d/yy h:mm
    37    #,##0 ;(#,##0)
    38    #,##0 ;[Red](#,##0)
    39    #,##0.00;(#,##0.00)
    40    #,##0.00;[Red](#,##0.00)
    45    mm:ss
    46    [h]:mm:ss
    47    mmss.0
    48    ##0.0E+0
    49    @
*/

// List of built-in cell style names
/*
    "normal"
    "comma"
    "currency"
    "percent"
    "comma0"
    "currency0"
    "note"
    "warningText"
    "title"
    "heading1"
    "heading2"
    "heading3"
    "heading4"
    "input"
    "output"
    "calculation"
    "checkCell"
    "linkedCell"
    "total"
    "good"
    "bad"
    "neutral"
    "accent1"
    "accent1pct20"
    "accent1pct40"
    "accent1pct60"
    "accent2"
    "accent2pct20"
    "accent2pct40"
    "accent2pct60"
    "accent3"
    "accent3pct20"
    "accent3pct40"
    "accent3pct60"
    "accent4"
    "accent4pct20"
    "accent4pct40"
    "accent4pct60"
    "accent5"
    "accent5pct20"
    "accent5pct40"
    "accent5pct60"
    "accent6"
    "accent6pct20"
    "accent6pct40"
    "accent6pct60"
    "explanatoryText"
*/

// The enumeration value indicating the portion of Cell Alignment
// in a cell format(XF) that is horizontal alignment.
enum HAlignEnum
{
    HALIGN_GENERAL,             // General Horizontal Alignment
    HALIGN_LEFT,                // Left Horizontal Alignment
    HALIGN_CENTER,              // Centered Horizontal Alignment
    HALIGN_RIGHT,               // Right Horizontal Alignment
    HALIGN_FILL,                // Fill
    HALIGN_JUSTIFY,             // Justify
    HALIGN_CENTERCONTINUOUS,    // Center Continuous Horizontal Alignment
    HALIGN_DISTRIBUTED          // Distributed Horizontal Alignment
};

// This enumeration value indicates the type of vertical alignment for a cell.
enum VAlignEnum
{
    VALIGN_BOTTOM,          // Aligned To Bottom
    VALIGN_CENTER,          // Centered Vertical Alignment
    VALIGN_TOP,             // Align Top
    VALIGN_JUSTIFY,         // Justified Vertically
    VALIGN_DISTRIBUTED      // Distributed Vertical Alignment
};

// The line style of a border in a cell(range).
enum BorderStyleEnum
{
    BORDER_NONE,               // None
    BORDER_THIN,               // Thin Border
    BORDER_MEDIUM,             // Medium Border
    BORDER_THICK,              // Thick Line Border
    BORDER_DOUBLE,             // Double Line
    BORDER_DASHED,             // Dashed
    BORDER_MEDIUMDASHED,       // Medium Dashed
    BORDER_DASHDOT,            // Dash Dot
    BORDER_MEDIUMDASHDOT,      // Medium Dash Dot
    BORDER_DASHDOTDOT,         // Dash Dot Dot
    BORDER_MEDIUMDASHDOTDOT,   // Medium Dash Dot Dot
    BORDER_SLANTDASHDOT,       // Slant Dash Dot
    BORDER_DOTTED,             // Dotted
    BORDER_HAIR                // Hairline Border
};

// Specify the border to retrieve
enum BorderIndexEnum
{
    BORDERINDEX_LEFT,
    BORDERINDEX_RIGHT,
    BORDERINDEX_TOP,
    BORDERINDEX_BOTTOM,
    BORDERINDEX_HORIZONTAL,
    BORDERINDEX_VERTICAL,
    BORDERINDEX_DIAGONAL_DOWN,
    BORDERINDEX_DIAGONAL_UP,

    BORDERINDEX_EDGE,
    BORDERINDEX_INSIDE
};

// The style of fill pattern being used for a cell format.
enum PatternTypeEnum
{
    PATTERNTYPE_NONE,
    
    PATTERNTYPE_SOLID,

    PATTERNTYPE_DARKGRAY,
    PATTERNTYPE_MEDIUMGRAY,
    PATTERNTYPE_LIGHTGRAY,
    PATTERNTYPE_GRAY125,
    PATTERNTYPE_GRAY0625,

    PATTERNTYPE_DARKHORIZONTAL,
    PATTERNTYPE_DARKVERTICAL,
    PATTERNTYPE_DARKUP,
    PATTERNTYPE_DARKDOWN,
    PATTERNTYPE_DARKTRELLIS,
    PATTERNTYPE_DARKGRID,

    PATTERNTYPE_LIGHTHORIZONTAL,
    PATTERNTYPE_LIGHTVERTICAL,
    PATTERNTYPE_LIGHTUP,
    PATTERNTYPE_LIGHTDOWN,
    PATTERNTYPE_LIGHTTRELLIS,
    PATTERNTYPE_LIGHTGRID
};

// An enumeration of the alignment of the contents of this run
// in relation to the default appearance of the run's text.
enum VertAlignRunEnum
{
    VALIGNRUN_BASELINE,     // Regular Vertical Positioning
    VALIGNRUN_SUBSCRIPT,    // Subscript
    VALIGNRUN_SUPERSCRIPT   // Superscript
};

//Represents the different types of possible underline formatting.
enum UnderlineEnum
{
    UNDERLINE_NONE,             // No underline.
    UNDERLINE_SINGLE,           // Single-line underlining under each character in the cell.
    UNDERLINE_SINGLEACCOUNTING, // Single-line accounting underlining under each character in the cell.
    UNDERLINE_DOUBLE,           // Double-line underlining under each character in the cell.
    UNDERLINE_DOUBLEACCOUNTING  // Double-line accounting underlining under each character in the cell.
};

// List of built-in table style names
/*
    ID          name
-----------------------------------
     0    "TableStyleLight1",
     1    "TableStyleLight2",
     2    "TableStyleLight3",
     3    "TableStyleLight4",
     4    "TableStyleLight5",
     5    "TableStyleLight6",
     6    "TableStyleLight7",
     7    "TableStyleLight8",
     8    "TableStyleLight9",
     9    "TableStyleLight10",
    10    "TableStyleLight11",
    11    "TableStyleLight12",
    12    "TableStyleLight13",
    13    "TableStyleLight14",
    14    "TableStyleLight15",
    15    "TableStyleLight16",
    16    "TableStyleLight17",
    17    "TableStyleLight18",
    18    "TableStyleLight19",
    19    "TableStyleLight20",
    20    "TableStyleLight21",
    21    "TableStyleMedium1",
    22    "TableStyleMedium2",
    23    "TableStyleMedium3",
    24    "TableStyleMedium4",
    25    "TableStyleMedium5",
    26    "TableStyleMedium6",
    27    "TableStyleMedium7",
    28    "TableStyleMedium8",
    29    "TableStyleMedium9",
    30    "TableStyleMedium10",
    31    "TableStyleMedium11",
    32    "TableStyleMedium12",
    33    "TableStyleMedium13",
    34    "TableStyleMedium14",
    35    "TableStyleMedium15",
    36    "TableStyleMedium16",
    37    "TableStyleMedium17",
    38    "TableStyleMedium18",
    39    "TableStyleMedium19",
    40    "TableStyleMedium20",
    41    "TableStyleMedium21",
    42    "TableStyleMedium22",
    43    "TableStyleMedium23",
    44    "TableStyleMedium24",
    45    "TableStyleMedium25",
    46    "TableStyleMedium26",
    47    "TableStyleMedium27",
    48    "TableStyleMedium28",
    49    "TableStyleDark1",
    50    "TableStyleDark2",
    51    "TableStyleDark3",
    52    "TableStyleDark4",
    53    "TableStyleDark5",
    54    "TableStyleDark6",
    55    "TableStyleDark7",
    56    "TableStyleDark8",
    57    "TableStyleDark9",
    58    "TableStyleDark10",
    59    "TableStyleDark11"
*/

// An enumeration that specifies what function is used to aggregate the
// data in a table column before it is displayed in the totals row.
enum TotalsFuncEnum
{
    TOTALS_FUNC_NONE,       // No total row.
    TOTALS_FUNC_SUM,        // Represents the arithmetic sum.
    TOTALS_FUNC_AVERAGE,    // Represents the arithmetic mean.
    TOTALS_FUNC_COUNT,      // Represents a count of the number of non-empty cells.
    TOTALS_FUNC_COUNT_NUMS, // Represents the number of cells that contain numbers.
    TOTALS_FUNC_CUSTOM,     // Represents the formula provided in totalsRowFormula.
    TOTALS_FUNC_MIN,        // Represents the smallest value.
    TOTALS_FUNC_MAX,        // Represents the largest value.
    TOTALS_FUNC_STDDEV,     // Represents the estimated standard deviation.
    TOTALS_FUNC_VAR         // Represents the estimated variance.
};

enum RowHeightUnitEnum
{
    ROWHEIGHT_POINTS,
    ROWHEIGHT_INCH,
    ROWHEIGHT_MM
};

enum ColWidthUnitEnum
{
    COLWIDTH_CHARS,
    COLWIDTH_INCH,
    COLWIDTH_MM
};

// for how DrawingML contents shall be resized when the rows andcolumns
// between its start and ending anchor(the from and to child elements)
// are resized, or have additional rows/columns inserted within them.
enum ImageAnchorEnum
{
    IMAGE_NOT_EXIST,
    IMAGE_ABSOLUTE,         // Do Not Move or Resize With Underlying Rows/Columns
    IMAGE_WITH_ONE_CELL,    // Move With Cells but Do Not Resize
    IMAGE_WITH_TWO_CELL     // Move and Resize With Anchor Cells
};

// Print orientation for sheet.
enum PageOrientEnum
{
    PAGEORIENT_DEFAULT,        // Orientation not specified, use the default.
    PAGEORIENT_LANDSCAPE,      // Landscape orientation.
    PAGEORIENT_PORTRAIT        // Portrait orientation.
};

// Specifies printed page order.
enum PrintOrderEnum
{
    PRINTORDER_DOWNTHENOVER, // Order pages vertically first, then move horizontally.
    PRINTORDER_OVERTHENDOWN, // Order pages horizontally first, then move vertically.
};

enum PaperUnitEnum
{
    PAPERUNIT_mm,
    PAPERUNIT_cm,
    PAPERUNIT_in,
    PAPERUNIT_pt,
    PAPERUNIT_pc,
    PAPERUNIT_pi
};

// These enumerations specify how cell comments shall be displayed for paper printing purposes.
enum PrintCMTEnum
{
    PRINTCMT_NONE,          // Do not print cell comments.
    PRINTCMT_ASDISPLAYED,   // Print cell comments as displayed.
    PRINTCMT_ATEND          // Print cell comments at end of document.
};

// This enumeration specifies how to display cells with errors when printing the worksheet.
enum PrintErrorEnum
{
    PRINTERROR_BLANK,       // Display cell errors as blank.
    PRINTERROR_DASH,        // Display cell errors as dashes.
    PRINTERROR_DISPLAYED,   // Display cell errors as displayed on screen.
    PRINTERROR_NA           // Display cell errors as #N/A.
};

/*
     ID          Paper
-----------------------------------------------------------
      1  =  Letter paper(8.5 in.by 11 in.)
      2  =  Letter small paper(8.5 in.by 11 in.)
      3  =  Tabloid paper(11 in.by 17 in.)
      4  =  Ledger paper(17 in.by 11 in.)
      5  =  Legal paper(8.5 in.by 14 in.)
      6  =  Statement paper(5.5 in.by 8.5 in.)
      7  =  Executive paper(7.25 in.by 10.5 in.)
      8  =  A3 paper(297 mm by 420 mm)
      9  =  A4 paper(210 mm by 297 mm)
     10  =  A4 small paper(210 mm by 297 mm)
     11  =  A5 paper(148 mm by 210 mm)
     12  =  B4 paper(250 mm by 353 mm)
     13  =  B5 paper(176 mm by 250 mm)
     14  =  Folio paper(8.5 in.by 13 in.)
     15  =  Quarto paper(215 mm by 275 mm)
     16  =  Standard paper(10 in.by 14 in.)
     17  =  Standard paper(11 in.by 17 in.)
     18  =  Note paper(8.5 in.by 11 in.)
     19  =  #9 envelope(3.875 in.by 8.875 in.)
     20  =  #10 envelope(4.125 in.by 9.5 in.)
     21  =  #11 envelope(4.5 in.by 10.375 in.)
     22  =  #12 envelope(4.75 in.by 11 in.)
     23  =  #14 envelope(5 in.by 11.5 in.)
     24  =  C paper(17 in.by 22 in.)
     25  =  D paper (22 in. by 34 in.)
     26  =  E paper (34 in. by 44 in.)
     27  =  DL envelope (110 mm by 220 mm)
     28  =  C5 envelope (162 mm by 229 mm)
     29  =  C3 envelope (324 mm by 458 mm)
     30  =  C4 envelope (229 mm by 324 mm)
     31  =  C6 envelope (114 mm by 162 mm)
     32  =  C65 envelope (114 mm by 229 mm)
     33  =  B4 envelope (250 mm by 353 mm)
     34  =  B5 envelope (176 mm by 250 mm)
     35  =  B6 envelope (176 mm by 125 mm)
     36  =  Italy envelope (110 mm by 230 mm)
     37  =  Monarch envelope (3.875 in. by 7.5 in.).
     38  =  6 3/4 envelope (3.625 in. by 6.5 in.)
     39  =  US standard fanfold (14.875 in. by 11 in.)
     40  =  German standard fanfold (8.5 in. by 12 in.)
     41  =  German legal fanfold (8.5 in. by 13 in.)
     42  =  ISO B4 (250 mm by 353 mm)
     43  =  Japanese double postcard (200 mm by 148 mm)
     44  =  Standard paper (9 in. by 11 in.)
     45  =  Standard paper (10 in. by 11 in.)
     46  =  Standard paper (15 in. by 11 in.)
     47  =  Invite envelope (220 mm by 220 mm)
     50  =  Letter extra paper (9.275 in. by 12 in.)
     51  =  Legal extra paper (9.275 in. by 15 in.)
     52  =  Tabloid extra paper (11.69 in. by 18 in.)
     53  =  A4 extra paper (236 mm by 322 mm)
     54  =  Letter transverse paper (8.275 in. by 11 in.)
     55  =  A4 transverse paper (210 mm by 297 mm)
     56  =  Letter extra transverse paper (9.275 in. by 12 in.)
     57  =  SuperA/SuperA/A4 paper (227 mm by 356 mm)
     58  =  SuperB/SuperB/A3 paper (305 mm by 487 mm)
     59  =  Letter plus paper (8.5 in. by 12.69 in.)
     60  =  A4 plus paper (210 mm by 330 mm)
     61  =  A5 transverse paper (148 mm by 210 mm)
     62  =  JIS B5 transverse paper (182 mm by 257 mm)
     63  =  A3 extra paper (322 mm by 445 mm)
     64  =  A5 extra paper (174 mm by 235 mm)
     65  =  ISO B5 extra paper (201 mm by 276 mm)
     66  =  A2 paper (420 mm by 594 mm)
     67  =  A3 transverse paper (297 mm by 420 mm)
     68  =  A3 extra transverse paper (322 mm by 445 mm)
     69  =  Japanese Double Postcard (200 mm x 148 mm)
     70  =  A6 (105 mm x 148 mm)
     71  =  Japanese Envelope Kaku #2
     72  =  Japanese Envelope Kaku #3
     73  =  Japanese Envelope Chou #3
     74  =  Japanese Envelope Chou #4
     75  =  Letter Rotated (11in x 8 1/2 11 in)
     76  =  A3 Rotated (420 mm x 297 mm)
     77  =  A4 Rotated (297 mm x 210 mm)
     78  =  A5 Rotated (210 mm x 148 mm)
     79  =  B4 (JIS) Rotated (364 mm x 257 mm)
     80  =  B5 (JIS) Rotated (257 mm x 182 mm)
     81  =  Japanese Postcard Rotated (148 mm x 100 mm)
     82  =  Double Japanese Postcard Rotated (148 mm x 200 mm)
     83  =  A6 Rotated (148 mm x 105 mm)
     84  =  Japanese Envelope Kaku #2 Rotated
     85  =  Japanese Envelope Kaku #3 Rotated
     86  =  Japanese Envelope Chou #3 Rotated
     87  =  Japanese Envelope Chou #4 Rotated
     88  =  B6 (JIS) (128 mm x 182 mm)
     89  =  B6 (JIS) Rotated (182 mm x 128 mm)
     90  =  (12 in x 11 in)
     91  =  Japanese Envelope You #4
     92  =  Japanese Envelope You #4 Rotated
     93  =  PRC 16K (146 mm x 215 mm)
     94  =  PRC 32K (97 mm x 151 mm)
     95  =  PRC 32K(Big) (97 mm x 151 mm)
     96  =  PRC Envelope #1 (102 mm x 165 mm)
     97  =  PRC Envelope #2 (102 mm x 176 mm)
     98  =  PRC Envelope #3 (125 mm x 176 mm)
     99  =  PRC Envelope #4 (110 mm x 208 mm)
    100  =  PRC Envelope #5 (110 mm x 220 mm)
    101  =  PRC Envelope #6 (120 mm x 230 mm)
    102  =  PRC Envelope #7 (160 mm x 230 mm)
    103  =  PRC Envelope #8 (120 mm x 309 mm)
    104  =  PRC Envelope #9 (229 mm x 324 mm)
    105  =  PRC Envelope #10 (324 mm x 458 mm)
    106  =  PRC 16K Rotated
    107  =  PRC 32K Rotated
    108  =  PRC 32K(Big) Rotated
    109  =  PRC Envelope #1 Rotated (165 mm x 102 mm)
    110  =  PRC Envelope #2 Rotated (176 mm x 102 mm)
    111  =  PRC Envelope #3 Rotated (176 mm x 125 mm)
    112  =  PRC Envelope #4 Rotated (208 mm x 110 mm)
    113  =  PRC Envelope #5 Rotated (220 mm x 110 mm)
    114  =  PRC Envelope #6 Rotated (230 mm x 120 mm)
    115  =  PRC Envelope #7 Rotated (230 mm x 160 mm)
    116  =  PRC Envelope #8 Rotated (309 mm x 120 mm)
    117  =  PRC Envelope #9 Rotated (324 mm x 229 mm)
    118  =  PRC Envelope #10 Rotated (458 mm x 324 mm)
*/

#endif // LIB_SPREADSHEET_ENUM_H

