#include "oosxl_module.h"


PyTypeObject RichtextType;

static void Richtext_dealloc(RichtextObject* self)
{
    Py_TYPE(self)->tp_free((PyObject*)self);
}

static PyObject* Richtext_segments_num(RichtextObject* self, PyObject* args)
{
    if (!check_handle(self->handle, "richtext"))
        return NULL;

    int result = rt_segments_num(self->handle);
    return PyLong_FromLong(result);
}

static PyObject* Richtext_plain_text(RichtextObject* self, PyObject* args)
{
    if (!check_handle(self->handle, "richtext"))
        return NULL;

    const char* result = rt_plain_text(self->handle);
    if (result == NULL)
    {
        Py_RETURN_NONE;
    }

    return PyUnicode_FromString(result);
}

static PyObject* Richtext_text(RichtextObject* self, PyObject* args)
{
    int index;

    if (!check_handle(self->handle, "richtext"))
        return NULL;

    if (!PyArg_ParseTuple(args, "i", &index))
    {
        return NULL;
    }

    StyleHandle style_handle = NULL;
    const char* result = rt_text(self->handle, index, &style_handle);
    if (result == NULL)
    {
        Py_RETURN_NONE;
    }

    PyObject* text_obj = PyUnicode_FromString(result);
    PyObject* style_obj = create_style_object(style_handle);

    return Py_BuildValue("(OO)", text_obj, style_obj);
}

static PyObject* Richtext_append(RichtextObject* self, PyObject* args)
{
    const char* s;
    PyObject* style_obj;

    if (!check_handle(self->handle, "richtext"))
        return NULL;

    if (!PyArg_ParseTuple(args, "sO", &s, &style_obj))
    {
        return NULL;
    }

    if (!PyObject_TypeCheck(style_obj, &StyleType))
    {
        PyErr_SetString(PyExc_TypeError, "Expected a Style object");
        return NULL;
    }

    StyleObject* style = (StyleObject*)style_obj;
    rt_append(self->handle, s, style->handle);
    Py_RETURN_NONE;
}

static PyObject* Richtext_insert(RichtextObject* self, PyObject* args)
{
    int index;
    const char* s;
    PyObject* style_obj;

    if (!check_handle(self->handle, "richtext"))
        return NULL;

    if (!PyArg_ParseTuple(args, "isO", &index, &s, &style_obj))
    {
        return NULL;
    }

    if (!PyObject_TypeCheck(style_obj, &StyleType))
    {
        PyErr_SetString(PyExc_TypeError, "Expected a Style object");
        return NULL;
    }

    StyleObject* style = (StyleObject*)style_obj;
    rt_insert(self->handle, index, s, style->handle);
    Py_RETURN_NONE;
}

static PyObject* Richtext_remove(RichtextObject* self, PyObject* args)
{
    int index;

    if (!check_handle(self->handle, "richtext"))
        return NULL;

    if (!PyArg_ParseTuple(args, "i", &index))
    {
        return NULL;
    }

    rt_remove(self->handle, index);
    Py_RETURN_NONE;
}

static PyObject* Richtext_modify(RichtextObject* self, PyObject* args)
{
    int index;
    const char* s;
    PyObject* style_obj;

    if (!check_handle(self->handle, "richtext"))
        return NULL;

    if (!PyArg_ParseTuple(args, "isO", &index, &s, &style_obj))
    {
        return NULL;
    }

    if (!PyObject_TypeCheck(style_obj, &StyleType))
    {
        PyErr_SetString(PyExc_TypeError, "Expected a Style object");
        return NULL;
    }

    StyleObject* style = (StyleObject*)style_obj;
    rt_modify(self->handle, index, s, style->handle);
    Py_RETURN_NONE;
}

static PyMethodDef Richtext_methods[] = {
    {"segments_num", (PyCFunction)Richtext_segments_num, METH_NOARGS, "Get number of segments"},
    {"plain_text", (PyCFunction)Richtext_plain_text, METH_NOARGS, "Get plain text content"},
    {"text", (PyCFunction)Richtext_text, METH_VARARGS, "Get text segment with style"},
    {"append", (PyCFunction)Richtext_append, METH_VARARGS, "Append text segment with style"},
    {"insert", (PyCFunction)Richtext_insert, METH_VARARGS, "Insert text segment with style"},
    {"remove", (PyCFunction)Richtext_remove, METH_VARARGS, "Remove text segment"},
    {"modify", (PyCFunction)Richtext_modify, METH_VARARGS, "Modify text segment"},
    {NULL} /* Sentinel */
};

PyTypeObject RichtextType = {
    PyVarObject_HEAD_INIT(NULL, 0)
        .tp_name = "oosxl.Richtext",
    .tp_doc = "Richtext object",
    .tp_basicsize = sizeof(RichtextObject),
    .tp_itemsize = 0,
    .tp_flags = Py_TPFLAGS_DEFAULT,
    .tp_new = PyType_GenericNew,
    .tp_dealloc = (destructor)Richtext_dealloc,
    .tp_methods = Richtext_methods,
};

PyObject* create_richtext_object(RichtextHandle handle)
{
    RichtextObject* obj = (RichtextObject*)PyObject_New(RichtextObject, &RichtextType);
    if (obj != NULL)
    {
        obj->handle = handle;
    }
    return (PyObject*)obj;
}
