import oosxl

def main():
    print(f"oosxl.Library Version: {oosxl.version()}")
    print(f"Run Message: {oosxl.run_message()}")
    
    # Test color coding
    print("\n=== Testing Color Encoding ===")
    rgb = oosxl.encoding_rgb(255, 128, 64)
    print(f"RGB(255, 128, 64) -> {rgb}")
    r, g, b = oosxl.decoding_rgb(rgb)
    print(f"{rgb} -> RGB({r}, {g}, {b})")
    
    argb = oosxl.encoding_argb(255, 128, 64, 32)
    print(f"ARGB(255, 128, 64, 32) -> {argb}")
    a, r, g, b = oosxl.decoding_argb(argb)
    print(f"{argb} -> ARGB({a}, {r}, {g}, {b})")
    
    # Test date and time encoding
    print("\n=== Testing Date/Time Encoding ===")
    date_serial = oosxl.encoding_date(2023, 12, 25)
    print(f"Date 2023-12-25 -> {date_serial}")
    year, month, day = oosxl.decoding_date(date_serial)
    print(f"{date_serial} -> {year}-{month}-{day}")
    
    time_serial = oosxl.encoding_time(14, 30, 45)
    print(f"Time 14:30:45 -> {time_serial}")
    hour, minute, second, msec = oosxl.decoding_time(time_serial)
    print(f"{time_serial} -> {hour:02d}:{minute:02d}:{second:02d}")
    
    # Testing A1 Coordinate Conversion
    print("\n=== Testing A1 Coordinate Conversion ===")
    row, col = oosxl.a1_to_rowcol("C5")
    print(f"A1 'C5' -> row={row}, col={col}")
    a1 = oosxl.rowcol_to_a1(10, 3)
    print(f"row=10, col=3 -> A1 '{a1}'")
    
    print("\n=== Creating New Document ===")
    doc = oosxl.create_document("test.xlsx")
    
    workbook = doc.get_workbook()
    
    workbook.set_date1904(False)
    
    worksheet = workbook.add_worksheet("Data")
    
    workbook.set_active_sheet(0)
    
    print("Creating styles...")
    header_style = workbook.make_normal_style()
    header_style.set_font("Arial", 14)
    header_style.set_bold(True)
    header_style.set_font_color(oosxl.encoding_rgb(255, 0, 0))
    
    number_style = workbook.make_normal_style()
    number_style.set_font("Arial", 12)
    number_style.set_align_h(3)
    
    print("Creating rich text...")
    rt = workbook.make_richtext()
    rt.append("Hello", header_style)
    rt.append(" World", number_style)
    
    print("Writing data...")
    worksheet.set_text(1, 1, "Product")
    worksheet.set_cell_style(1, 1, header_style)
    worksheet.set_text(1, 2, "Quantity")
    worksheet.set_cell_style(1, 2, header_style)
    worksheet.set_text(1, 3, "Price")
    worksheet.set_cell_style(1, 3, header_style)
    worksheet.set_richtext(1, 4, rt)
    
    data = [
        ("Apple", 10, 1.50),
        ("Banana", 15, 0.75),
        ("Orange", 8, 1.20),
        ("Grapes", 12, 2.50)
    ]
    
    for i, (product, quantity, price) in enumerate(data, 1):
        worksheet.set_text(i + 2, 1, product)
        worksheet.set_num(i + 2, 2, quantity)
        worksheet.set_num(i + 2, 3, price)
        worksheet.set_cell_style(i + 2, 3, number_style)
    
    worksheet.set_custom_col_width(1, 1, 15, 0)
    worksheet.set_custom_col_width(2, 4, 20, 0)
    
    print("Adding list object...")
    lo = worksheet.add_tabledef("SalesData", 2, 1, 7, 4)
    lo.set_display_name("Sales Data")
    lo.set_col_name(0, "Product Name")
    lo.set_col_name(1, "Quantity Sold")
    lo.set_col_name(2, "Unit Price")
    
    print(f"\n=== Worksheet Info ===")
    print(f"Min Row: {worksheet.min_row()}")
    print(f"Max Row: {worksheet.max_row()}")
    print(f"Min Col: {worksheet.min_col()}")
    print(f"Max Col: {worksheet.max_col()}")
    print(f"List Object Count: {worksheet.tabledef_count()}")
    
    print("\n=== Reading Back Data ===")
    for i in range(1, len(data) + 2):
        for j in range(1, 4):
            cell_type = worksheet.cell_type(i, j)
            if cell_type == 2:
                value = worksheet.get_text(i, j)
            elif cell_type == 1:
                value = worksheet.get_num(i, j)
            else:
                value = "Unknown"
            print(f"Cell ({i},{j}): {value}")
    
    print("\nSaving document...")
    doc.save("test.xlsx")
    print("Document saved successfully!")

if __name__ == "__main__":
    main()
