/**
 * worksheet.hxx
 *
 *      Copyright (c) 2025 CHBXUE. All rights reserved.
 *
 * THIS FILE AND THE SOFTWARE CONTAINED HEREIN IS PROVIDED 'AS IS' AND
 * COMES WITH NO WARRANTIES OF ANY KIND.
 */

#ifndef WORKSHEET_INTERFACE_H
#define WORKSHEET_INTERFACE_H

#include "port.h"
#include "enum.h"


namespace oo {

struct iStyle;
struct iRichtext;
struct iTabledef;

struct iWorksheet
{
    virtual             int  min_row() const = 0;
    virtual             int  min_col() const = 0;
    virtual             int  max_row() const = 0;
    virtual             int  max_col() const = 0;

    virtual      const char* name() const = 0;
    virtual            bool  set_name(const char* newname) = 0;
    virtual   SheetTypeEnum  sheet_type() const = 0;
    virtual  SheetStateEnum  get_sheet_state() const = 0;

    virtual    CellTypeEnum  cell_type(int row, int col) const = 0;
    virtual            bool  contains_formula(int row, int col) const = 0;
    virtual            bool  contains_richtext(int row, int col) const = 0;
    virtual          double  get_num(int row, int col) const = 0;
    virtual      const char* get_text(int row, int col) const = 0;
    virtual            bool  get_boolean(int row, int col) const = 0;
    virtual          double  get_datetime(int row, int col) const = 0;
    virtual       iRichtext* get_richtext(int row, int col) const = 0;
    virtual      const char* get_formula_expr(int row, int col) const = 0;
    virtual   ErrorCodeEnum  get_error(int row, int col) const = 0;
    virtual            bool  set_num(int row, int col, double val) = 0;
    virtual            bool  set_text(int row, int col, const char* val) = 0;
    virtual            bool  set_boolean(int row, int col, bool b) = 0;
    virtual            bool  set_datetime(int row, int col, double dt) = 0;
    virtual            bool  set_richtext(int row, int col, iRichtext* rt) = 0;
    virtual            bool  set_error(int row, int col, ErrorCodeEnum code) = 0;
    virtual            bool  set_formula(int row, int col, const char* expr) = 0;
    virtual            bool  set_formula_num(int row, int col, double val) = 0;
    virtual            bool  set_formula_string(int row, int col, const char* s) = 0;
    virtual            bool  set_formula_boolean(int row, int col, bool b) = 0;
    virtual            bool  set_formula_datetime(int row, int col, double dt) = 0;
    virtual            bool  set_formula_error(int row, int col, ErrorCodeEnum code) = 0;
    virtual            void  clear_contents(int beginRow, int beginCol, int end_row, int end_col) = 0;

    virtual            void  set_cell_style(int row, int col, iStyle* style) = 0;
    virtual          iStyle* cell_style(int row, int col) const = 0;
    virtual          iStyle* get_style(int row, int col) const = 0;
    virtual            void  set_range_style(int start_row, int start_col, int stop_row, int stop_col, iStyle* style) = 0;

    virtual            bool  is_default_row_height(int start, int end = 0) const = 0;
    virtual            bool  get_custom_row_height(int start, int end, double* height, RowHeightUnitEnum unit = ROWHEIGHT_POINTS) const = 0;
    virtual            void  set_custom_row_height(int start, int end, double height, RowHeightUnitEnum unit = ROWHEIGHT_POINTS) = 0;
    virtual            bool  is_default_col_width(int start, int end = 0) const = 0;
    virtual            bool  get_custom_col_width(int start, int end, double* width, ColWidthUnitEnum unit = COLWIDTH_CHARS) const = 0;
    virtual            void  set_custom_col_width(int start, int end, double width, ColWidthUnitEnum unit = COLWIDTH_CHARS) = 0;

    virtual          iStyle* row_style(int row) = 0;
    virtual            void  remove_row_style(int row) = 0;
    virtual            void  set_row_style(int start_row, int stop_row, iStyle* style) = 0;
    virtual          iStyle* col_style(int col) = 0;
    virtual            void  remove_col_style(int col) = 0;
    virtual            void  set_col_style(int start_col, int stop_col, iStyle* style) = 0;

    virtual            bool  row_hidden(int row) const = 0;
    virtual            void  set_row_hidden(int start_row, int stop_row, bool hidden = true) = 0;
    virtual            bool  col_hidden(int col) const = 0;
    virtual            void  set_col_hidden(int start_col, int stop_col, bool hidden = true) = 0;

    virtual            bool  insert_row(int row, int count = 1) = 0;
    virtual            bool  delete_row(int begin, int end = 0) = 0;
    virtual            bool  insert_col(int col, int count = 1) = 0;
    virtual            bool  delete_col(int begin, int end = 0) = 0;

    virtual            bool  shift_range_down(int start_col, int end_col, int start_row, int count = 1) = 0;
    virtual            bool  shift_range_up(int start_col, int end_col, int start_row, int count = 1) = 0;
    virtual            bool  shift_range_right(int start_row, int end_row, int start_col, int count = 1) = 0;
    virtual            bool  shift_range_left(int start_row, int end_row, int start_col, int count = 1) = 0;

    virtual            void  set_bg_picture(const char* file) = 0;
    virtual             int  add_cell_image(const char* file, int start_row, int start_col, int end_row, int end_col) = 0;
    virtual             int  add_image_with_two_cells(const char* file, int row1, int col1, int row2, int col2,
                                                      int offset_x1 = 0, int offset_y1 = 0, int offset_x2 = 0, int offset_y2 = 0) = 0;
    virtual             int  add_image_with_one_cell(const char* file, int row, int col, int width = 0, int height = 0,
                                                     double scale = 1.0, int offset_x = 0, int offset_y = 0) = 0;
    virtual             int  add_absolute_image(const char* file, int x, int y, int width = 0, int height = 0, double scale = 1.0) = 0;
    virtual             int  image_num() const = 0;
    virtual             int  get_image_id(int index) const = 0;
    virtual ImageAnchorEnum  image_anchor_type(int image_id) const = 0;
    virtual            bool  get_anchored_cell(int image_id, int* row = 0, int* col = 0) = 0;
    virtual            bool  get_absolute_coord(int image_id, int* x = 0, int* y = 0) = 0;
    virtual            void  delete_image(int image_id) = 0;
    virtual            void  remove_image(int row, int col) = 0;
    virtual            void  remove_absolute_image(int x, int y) = 0;

    virtual            bool  first_data_cell(int* row, int* col) const = 0;
    virtual            bool  last_data_cell(int* row, int* col) const = 0;

    virtual             int  merged_num() const = 0;
    virtual            bool  merged_range(int index, int* start_row, int* start_col, int* stop_row, int* stop_col) const = 0;
    virtual            bool  merge(int start_row, int start_col, int stop_row, int stop_col) = 0;
    virtual            bool  unmerge(int row, int col) = 0;

    virtual             int  tabledef_count() const = 0;
    virtual      const char* get_tabledef_name(int index) const = 0;
    virtual       iTabledef* set_tabledef_name(int index, const char* newName) = 0;
    virtual       iTabledef* get_tabledef(int index) const = 0;
    virtual       iTabledef* get_tabledef_by_name(const char* name) const = 0;
    virtual       iTabledef* add_tabledef(const char* name, int beginRow, int beginCol, int end_row, int end_col) = 0;
    virtual            bool  delete_tabledef(const char* name) = 0;

    virtual            void  group_rows(int beginRow, int end_row, bool collapsed = false) = 0;
    virtual            void  ungroup_rows(int beginRow, int end_row) = 0;
    virtual            void  group_cols(int beginCol, int end_col, bool collapsed = true) = 0;
    virtual            void  ungroup_cols(int beginCol, int end_col) = 0;
    virtual             int  row_outline_level(int row, bool* collapsed = 0) const = 0;
    virtual             int  col_outline_level(int col, bool* collapsed = 0) const = 0;
    virtual             int  max_row_outline_level() const = 0;
    virtual             int  max_col_outline_level() const = 0;
    virtual            void  delete_row_outline() = 0;
    virtual            void  delete_col_outline() = 0;
    virtual            bool  group_summary_below() const = 0;
    virtual            void  set_group_summary_below(bool below) = 0;
    virtual            bool  group_summary_right() const = 0;
    virtual            void  set_group_summary_right(bool right) = 0;

    virtual            bool  add_hyperlink(const char* hyperlink, int start_row, int start_col, int end_row, int end_col) = 0;
    virtual             int  hyperlink_num() const = 0;
    virtual             int  hyperlink_index(int row, int col) const = 0;
    virtual      const char* hyperlink(int index, int* start_row, int* start_col, int* end_row, int* end_col) = 0;
    virtual            bool  delete_hyperlink(int index) = 0;

    virtual            bool  is_protected() const = 0;
    virtual            bool  protect(SheetProtectEnum item = PROTECT_DEFAULT, const char* password = 0) = 0;
    virtual            void  unprotect() = 0;
    virtual            bool  protect_range(int start_row, int start_col, int end_row, int end_col) = 0;
    virtual           bool   set_editable_range(int start_row, int start_col, int end_row, int end_col) = 0;
    virtual            int   editable_range_count() const = 0;
    virtual     const char*  get_editable_range(int index, int* start_row = 0, int* start_col = 0, int* stop_row = 0, int* stop_col = 0) const = 0;
    virtual           bool   delete_editable_range(int index) = 0;

    virtual             int  zoom() const = 0;
    virtual            void  set_zoom(int zoom) = 0;
    virtual            bool  right_to_left() const = 0;
    virtual            void  set_right_to_left(bool right_to_left = true) = 0;
    virtual            void  topleft_cell(int* row, int* col) const = 0;
    virtual            void  set_topleft_cell(int row, int col) = 0;
    virtual            void  active_cell(int* row, int* col) const = 0;
    virtual            void  set_active_cell(int row, int col) = 0;
    virtual            void  split_cell(int* row, int* col) const = 0;
    virtual            void  set_split_cell(int row, int col) = 0;
    virtual            bool  display_gridlines() const = 0;
    virtual            void  set_display_gridlines(bool show = true) = 0;

    virtual             int  row_break_count() const = 0;
    virtual             int  row_break(int index) const = 0;
    virtual            bool  set_row_break(int row, bool pageBreak = true) = 0;
    virtual             int  col_break_count() const = 0;
    virtual             int  col_break(int index) const = 0;
    virtual            bool  set_col_break(int col, bool pageBreak = true) = 0;

    virtual            bool  print_gridlines() const = 0;
    virtual            void  set_print_gridlines(bool print = true) = 0;
    virtual            bool  print_headings() const = 0;
    virtual            void  set_print_headings(bool print = false) = 0;
    virtual            bool  h_centered_printing() const = 0;
    virtual            void  set_h_centered_printing(bool centered = true) = 0;
    virtual            bool  v_centered_printing() const = 0;
    virtual            void  set_v_centered_printing(bool centered = true) = 0;

    virtual          double  left_margin() const = 0;
    virtual            void  set_left_margin(double value) = 0;
    virtual          double  right_margin() const = 0;
    virtual            void  set_right_margin(double value) = 0;
    virtual          double  top_margin() const = 0;
    virtual            void  set_top_margin(double value) = 0;
    virtual          double  bottom_margin() const = 0;
    virtual            void  set_bottom_margin(double value) = 0;
    virtual          double  header_margin() const = 0;
    virtual            void  set_header_margin(double value) = 0;
    virtual          double  footer_margin() const = 0;
    virtual            void  set_footer_margin(double value) = 0;

    virtual             int  paper() const = 0;
    virtual            bool  set_paper(int paper_code) = 0;
    virtual             int  print_scale() const = 0;
    virtual            void  set_print_scale(int zoom) = 0;
    virtual  PageOrientEnum  page_orientation() const = 0;
    virtual            void  set_page_orientation(PageOrientEnum orientation) = 0;
    virtual  PrintErrorEnum  print_errors() const = 0;
    virtual            void  set_print_errors(PrintErrorEnum way) = 0;
    virtual             int  fit_page_height() const = 0;
    virtual            void  set_fit_page_height(int num) = 0;
    virtual             int  fit_page_width() const = 0;
    virtual            void  set_fit_page_width(int num) = 0;

    virtual      const char* page_header() const = 0;
    virtual            void  set_page_header(const char* header) = 0;
    virtual      const char* page_footer() const = 0;
    virtual            void  set_page_footer(const char* footer) = 0;

    virtual ~iWorksheet() {}
};

}

#endif // WORKSHEET_INTERFACE_H

